#! /bin/sh
#----------------------------------------------------------------------------
# Get the average round trips per second
#
# Parameters:
# $1 : Console log name
# $2 : Mode
#      0: Calculate average of all test shots
#      1: Calculate the average of the remaining test shots, excluding
#        the smallest and largest times (valid when the number of shots is
#        3 or more)
#
# History
#
# Date        Create/Update           Description
# 2022.02.21  Pulsar Integration Inc. Initial Version
#----------------------------------------------------------------------------
function fnGetAverageRoundTripsPerSec
{
  ConsoleLog=$1
  Mode=$2

  awk '
  BEGIN {
    minimum_rtps=0
    maximum_rtps=0
    cumulative_rtps=0
    test_count=0
  }
  /Round Trips/ {
    rtps = $3

    if ( minimum_rtps == 0 || minimum_rtps > rtps ) {
      minimum_rtps = rtps
    }

    if ( maximum_rtps == 0 || maximum_rtps < rtps ) {
      maximum_rtps = rtps
    }

    #printf( "minimum_rtps: %f, rtps: %f\n", minimum_rtps, rtps)
    #printf( "maximum_rtps: %f, rtps: %f\n", maximum_rtps, rtps)

    cumulative_rtps += rtps
    test_count += 1
  }
  END {
    #printf( "cumulative: %f ", cumulative_rtps )
    printf( "cumulative: %-8.3f ", cumulative_rtps )

    if ( Mode == 1 && test_count >= 3 ) {
      effective_rtps = cumulative_rtps - ( minimum_rtps + maximum_rtps )
      test_count -= 2
      #printf( "minimum: %f maximum: %f ", minimum_rtps, maximum_rtps )
      printf( "minimum: %-8.3f maximum: %-8.3f ", minimum_rtps, maximum_rtps )
    }
    else {
      effective_rtps = cumulative_rtps
    }
    #print
    #print cumulative_rtps
    #print effective_rtps
    #print test_count
    average_rtps = effective_rtps / test_count
    #printf( "average: %f rtps\n", average_rtps )
    printf( "average: %-8.3f rtps\n", average_rtps )

  }' "Mode=$Mode" $ConsoleLog

  return $?
}

#------------------------------------------------------
# Calculate Elapsed Time(sec.millisec)
#
# Parameters:
# $1 : Start time string
# $2 : Stop time string
#      format "YY/MM/DD HH:MM:SS.nnnnnn"
# $3 : Start time(ms)
# $4 : Stop time(ms)
# $5 : Number of total messages
#
# History
#
# Date        Create/Update           Description
# 2022.02.21  Pulsar Integration Inc. Initial Version
#------------------------------------------------------
function fnCalcElapsedTime
{
  STRTIME=`echo $1 | sed "s/_/ /" | cut -b 1-21`
  STPTIME=`echo $2 | sed "s/_/ /" | cut -b 1-21`
  echo "Start Time: $STRTIME Stop Time: $STPTIME"

  echo "$3 $4 $5" |
  awk '{
    ELAPSE =  $2 - $1
    printf( "Elapsed Time: %-8.3f\n", ELAPSE/1000 )

    MSGS = $3*1
    printf( "Total Number of messages: %d\n", MSGS )
    printf( "Round Trips/sec: %-8.3f\n", MSGS/(ELAPSE/1000) )
  }'
}

#------------------------------------------------------
# Get Unix Time(millisec)
#
# Parameters:
# $1 : Time string
#      format "YY/MM/DD HH:MM:SS.nnnnnn"
#
# History
#
# Date        Create/Update           Description
# 2022.02.21  Pulsar Integration Inc. Initial Version
#------------------------------------------------------
function fnGetUnixTime
{
  DT="20$1"

  echo "$DT" |
  awk '{

#         12345678901234567890123456
# format "2099/12/31 12:59:59.999999"

    Y = substr($1,1,4)*1;
    M = substr($1,6,2)*1;
    D = substr($1,9,2)*1;
    h = substr($1,12,2)*1;
    m = substr($1,15,2)*1;
    s = substr($1,18,2)*1;
    i = substr($1,21,3)*1;

    if (M<3) {M+=12; Y--;}

    SEC=(365*Y+int(Y/4)-int(Y/100)+int(Y/400)+int(306*(M+1)/10)-428+D-719163)*86400+(h*3600)+(m*60)+s;

    MS=SEC*1000+i
    print MS
  }'
}

#------------------------------------------------------
# Get Elapsed Time(9.99999999999999sec) for Thread
#
# Parameters:
# $1 : Target log name
# $2 : Number of total messages
# $3 : Number of connections
#
# History
#
# Date        Create/Update           Description
# 2022.02.21  Pulsar Integration Inc. Initial Version
#------------------------------------------------------
function fnGetElapsedTime_Thread
{
  LOGFILE=$1
  MESSAGES=$2

  START_TIME_MS=0
  STOP_TIME_MS=0

  START_TIME=""
  STOP_TIME=""

  START_TIME=`grep "^\[../../.. " $LOGFILE | head -1 | awk -F" |]" '{ print $1 "_" $2 }' | sed "s/\[//"`
  STOP_TIME=`grep "^\[../../.. " $LOGFILE | tail -1 | head -1 | awk -F" |]" '{ print $1 "_"  $2 }' | sed "s/\[//"`
  START_TIME_MS=$(fnGetUnixTime $START_TIME)
  STOP_TIME_MS=$(fnGetUnixTime $STOP_TIME)

#print "start time st: $START_TIME stop time st: $STOP_TIME"
#print "start time ms: $START_TIME_MS stop time ms: $STOP_TIME_MS"
  fnCalcElapsedTime $START_TIME $STOP_TIME $START_TIME_MS $STOP_TIME_MS $MESSAGES
}

#------------------------------------------------------
# Get Elapsed Time(sec.millisec)
#
# Parameters:
# $1 : Target log name prefix
# $2 : Number of total  messages
#
# History
#
# Date        Create/Update           Description
# 2022.02.21  Pulsar Integration Inc. Initial Version
#------------------------------------------------------
function fnGetElapsedTime
{
  LOGFILE_PREFIX=$1
  MESSAGES=$2

  START_TIME_MS=0
  STOP_TIME_MS=0

  START_TIME_STR=""
  STOP_TIME_STR=""

  for LOGFILE in ${LOGFILE_PREFIX}_*
  do
    START_TIME=`grep "^\[../../.. " $LOGFILE | head -1 | awk -F" |]" '{ print $1 "_" $2 }' | sed "s/\[//"`
    STOP_TIME=`grep "^\[../../.. " $LOGFILE | tail -1 | head -1 | awk -F" |]" '{ print $1 "_"  $2 }' | sed "s/\[//"`
    START_TIME_MS_TMP=$(fnGetUnixTime $START_TIME)
    STOP_TIME_MS_TMP=$(fnGetUnixTime $STOP_TIME)
    if ((START_TIME_MS == 0))
    then
      START_TIME_MS=$START_TIME_MS_TMP
      START_TIME_STR=$START_TIME
    elif ((START_TIME_MS > START_TIME_MS_TMP))
    then
      START_TIME_MS=$START_TIME_MS_TMP
      START_TIME_STR=$START_TIME
    fi

    if ((STOP_TIME_MS == 0))
    then
      STOP_TIME_MS=$STOP_TIME_MS_TMP
      STOP_TIME_STR=$STOP_TIME
    elif ((STOP_TIME_MS < STOP_TIME_MS_TMP))
    then
      STOP_TIME_MS=$STOP_TIME_MS_TMP
      STOP_TIME_STR=$STOP_TIME
    fi
  done
#print "start time st: $START_TIME_STR stop time st: $STOP_TIME_STR"
#print "start time ms: $START_TIME_MS stop time ms: $STOP_TIME_MS"
  fnCalcElapsedTime $START_TIME_STR $STOP_TIME_STR $START_TIME_MS $STOP_TIME_MS $MESSAGES
}

