# Windows PwowerShell
#---------------------------------------------------------
#
# RR-2 MQI Client Bindings(Thread) (for Requester)
#
# Parameters:
# $1 : Queue manager name
# $2 : Number of connections
# $3 : Messages
# $4 : Message Bytes
# $5 : Log Directory
# $6 : Workload
# $7 : MQ Constant
# $8 : Data File
#
# Environments(./runpfmts.bat):
# $INCL_CON_TIME : Include MQCONN/MQDISC time in measurement,
#                  0:not include, 1:include
# $RCV_BUFF_SIZE : Receive buffer size
# $CON_RETRY     : The number of connection retry
# $CON_INTERVAL  : Connection retry interval
# $RR2_RequestQ  : Request queue name
# $RR2_ReplyQ    : Reply queue name
# $MQSSLKEYR     : The location of the key repository(Used by MQ Product)
# $CERTLABL      : Certificate Label
# $SSLCIPH       : Cipher Spec
#
# History
#
# Date        Create/Update           Description
# 2022.02.21  Pulsar Integration Inc. Initial Version
#
#---------------------------------------------------------
param(
    [string]$TARGET_QMGR = "",
    [int]$Connections = 0,
    [int]$Messages = 0,
    [int]$Bytes = 0,
    [string]$LogDirPre = "",
    [string]$Workload = "",
    [string]$MQ_Constant = "",
    [string]$DataFile = ""
)

# return code
[int]$RET_OK=0
[int]$RET_NG=1

$USAGE = $myInvocation.MyCommand.name + " <Qmgr> <Connections> <Messages> <Bytes> <Log Directory> <Workload> <MQ Constant> <Data File>"

if ($PSBoundParameters.Count -lt 6)
{
    Write-output "" "Usage: $USAGE" "" ""

    $usage_document = @"
    RR-2 MQI Client Bindings(Thread) (for Requester)

    Parameters:
    `$1 : Queue manager name
    `$2 : Number of connections
    `$3 : Number of messages
    `$4 : Message Bytes
    `$5 : Log Directory
    `$6 : Workload
    `$7 : MQ Constant (ex. MQPER_PERSISTENT, MQPER_NOT_PERSISTENT)
    `$8 : Data File (Input message file) (Option)
         If not specified, "<workload name>" is used as the message.

    Environments(./runpfmts.bat):
    `$INCL_CON_TIME : Include MQCONN/MQDISC time in measurement,
                     0:not include, 1:include
    `$RCV_BUFF_SIZE : Receive buffer size
    `$CON_RETRY     : The number of connection retry
    `$CON_INTERVAL  : Connection retry interval
    `$RR2_RequestQ  : Request queue name
    `$RR2_ReplyQ    : Reply queue name
    `$MQSSLKEYR     : The location of the key repository(Used by MQ Product)
    `$CERTLABL      : Certification Label
    `$SSLCIPH       : Cipher Specification

"@

    Write-output $usage_document

    exit $RET_NG
}

# import library
. ./lib/libpfmts.ps1

# source environments for performance test
fnSourceEnvironments

[string]$RR2_RequestQ=${env:RR2_RequestQ}
[string]$RR2_ReplyQ=${env:RR2_ReplyQ}
[string]$CONNAME=${env:CONNAME}
[string]$CHANNEL=${env:CHANNEL}
[int]$CON_RETRY=${env:CON_RETRY}
[int]$CON_INTERVAL=${env:CON_INTERVAL}
[int]$RCV_BUFF_SIZE=${env:RCV_BUFF_SIZE}
[int]$INCL_CON_TIME=${env:INCL_CON_TIME}
[string]$CERTLABL=${env:CERTLABL}
[string]$SSLCIPH=${env:SSLCIPH}

$TRM = ""
if ($INCL_CON_TIME -eq 1)
{
    $TRM = "-tr"
}

#Write-Host "MQSSLKEYR: ${env:MQSSLKEYR}"

$CPH = ""
#if (Test-Path variable:SSLCIPH )
if ($SSLCIPH -ne "" )
{
    #Write-Host "SSLCIPH: $SSLCIPH"
    $CPH = "-cs $SSLCIPH"
}

$CLB = ""
#if (Test-Path variable:CERTLABL )
if ( $CERTLABL -ne "" )
{
    #Write-Host "CERTLABL: $CERTLABL"
    $CLB = "-cl $CERTLABL MQCD_VERSION_11"
}


$dateTime = Get-Date -Format "yyyyMMdd_HHmmss"
$LogDir = "${LogDirPre}\${Workload}_${Bytes}_${Connections}_${TARGET_QMGR}_${dateTime}"

mkdir $LogDir

$LogNamePre = "${LogDir}\req"
$StderrLogNamePre = "${LogDir}\stderr_req"

$logName = "${LogNamePre}_${dateTime}.log"
$stderrLogName = "${StderrLogNamePre}_${dateTime}.log"

Write-Host ""
Write-Host "TARGET_QMGR: $TARGET_QMGR"
Write-Host "Connections: $Connections Messages: $Messages Bytes: $Bytes LogDirPre: $LogDirPre Workload: $Workload MQ_Constant: $MQ_Constant DataFile: $DataFile"

# Requester
# secondary side: MQGMO_ACCEPT_TRUNCATED_MSG
Write-Host "Start Requester Program, Put $RR2_RequestQ, Get $RR2_ReplyQ"
if ($DataFile -eq "")
{
    $ArgumentList = "-qm $TARGET_QMGR -q $RR2_RequestQ -x $CONNAME -ch $CHANNEL -m $Workload -iq $RR2_ReplyQ -rq $RR2_ReplyQ MQMT_REQUEST $MQ_CONSTANT MQPMO_NO_SYNCPOINT -ni $Connections -n $Messages -l $Bytes -cr $CON_RETRY -wp $CON_INTERVAL $TRM $CPH $CLB -ss -sz $RCV_BUFF_SIZE MQGMO_WAIT MQWI_UNLIMITED MQGMO_NO_SYNCPOINT MQGMO_ACCEPT_TRUNCATED_MSG -mc"
}
else
{
    $ArgumentList = "-qm $TARGET_QMGR -q $RR2_RequestQ -x $CONNAME -ch $CHANNEL -f $DataFile -iq $RR2_ReplyQ -rq $RR2_ReplyQ MQMT_REQUEST $MQ_CONSTANT MQPMO_NO_SYNCPOINT -ni $Connections -n $Messages -l $Bytes -cr $CON_RETRY -wp $CON_INTERVAL $TRM $CPH $CLB -ss -sz $RCV_BUFF_SIZE MQGMO_WAIT MQWI_UNLIMITED MQGMO_NO_SYNCPOINT MQGMO_ACCEPT_TRUNCATED_MSG -mc"
}

Write-Host "mqpgfc $ArgumentList"
$WaitProcess = Start-Process -FilePath mqpgfc.exe -RedirectStandardOutput $logName -RedirectStandardError $stderrLogName -PassThru -NoNewWindow -ArgumentList $ArgumentList
try {
    do {
        $process = Get-Process -Id $WaitProcess.Id -ErrorAction stop | Format-Table
        $process
        Start-Sleep 5
    }
    while (1)
}
catch {
    Write-Output "The end of the test command has been detected."
}

Write-Host ""
fnGetElapsedTime_Thread -logName "$logName" -MESSAGES $Messages -CONNECTIONS $Connections | tee-object ${LogDirPre}/console.log -Append

exit $RET_OK
