# Windows PwowerShell
#---------------------------------------------------------
# Create and configure QMGR
#
# Arguments:
# $1 or -m : Queue manager name(mandatory)
# $2 or -f ; Source mqsc file name(option)
# Read <QMGR>.mqsc.src and <QMGR>.ini from current directory
# Write <QMGR>.mqsc
#
# History
#
# Date        Create/Update           Description
# 2022.02.21  Pulsar Integration Inc. Initial Version
#---------------------------------------------------------
Param(
    [String]$m = "",
    [String]$f = ""
)

# return code
[int]$RET_OK=0
[int]$RET_NG=1

# start up wait time
[int]$QMGR_STARTUP_TIME=5
[int]$CMDSVR_STARTUP_TIME=3

#---------------------------------------------
# Functions for Command Execution
#---------------------------------------------
function fnExecCmd
{
    $EXECCMD=$args[0]
    Write-Host ""
    Write-Host "Command: $EXECCMD"
    Write-Host ""

    Invoke-Expression "$EXECCMD" | Write-Host

    return $RET_OK
}

#-----------------------------------------------------
# Functions for Command Execution with Confirmation
#-----------------------------------------------------
function fnExecCmdConfirm
{
    $EXECCMD=$args[0]
    Write-Host ""
    Write-Host "Command: $EXECCMD"
    Write-Host ""

    $ANS = Read-host "Are you sure you want to run above command? [y/n]"
    if ( $ANS -ne "Y" )
    {
        Write-Host "Operation canceld."
        Write-Host ""
        return $RET_NG
    }

    Invoke-Expression "$EXECCMD" | Write-Host

    return $RET_OK
}

#---------------------------------------------
# Functions for Creating QMGR
#
# input:
# $TARGET_QMGR: Queue Manager Name
#---------------------------------------------
function fnCreateQmgr
{
    dspmq -m $TARGET_QMGR 2>&1 > $null
    if ($? -eq $True)
    {
      Write-Host ""
      Write-Host "QMGR($TARGET_QMGR) already exist."
      Write-Host ""

      $ANS = Read-host "Do you want to continue processing? [y/n]"
      if ( $ANS -ne "Y" )
      {
          Write-Host "Operation canceld."
          Write-Host ""
          return $RET_NG
      }
    }
    else
    {
        $retcd = fnExecCmdConfirm "crtmqm $TARGET_QMGR"
        return $retcd
    }
    return $RET_OK
}

#---------------------------------------------
# Functions for Starting qmgr
#
# input:
# $TARGET_QMGR: Queue Manager Name
#---------------------------------------------
function fnStartQmgr
{
    $EXECCMD="strmqm $TARGET_QMGR"

    Write-Host "Starting Qmgr: $TARGET_QMGR"
    Write-Host ""

    fnExecCmd "$EXECCMD"

    return $RET_OK
}

#------------------------------------------------------------------
# Functions for replacing configuration parameters
#
# input:
# $TARGET_QMGR: Queue Manager Name
# ${TARGET_QMGR}.mqsc.src: MQSC source file
#
# output:
# ${TARGET_QMGR}.mqsc: MQSC file
#------------------------------------------------------------------
function fnReplaceParameters
{
    Param(
        [String]$sourceFile = "${TARGET_QMGR}.mqsc.src"
    )

    $targetFile = "${TARGET_QMGR}.mqsc"
    $paramFile = "${TARGET_QMGR}.ini"

    if (!(Test-Path ./$sourceFile))
    {
      Write-Host "Cannot find source file ./$sourcefile"
      return $RET_NG
    }

    if (!(Test-Path ./$paramFile))
    {
      Write-Host "Cannot find parameter file ./$paramFile"
      return $RET_NG
    }

    copy ./$sourceFile ./$targetFile

    $prm_f = (Get-Content ./$paramFile | select-string "^\s*$" -notmatch | select-string "^\s*#.*$" -notmatch) -as [string[]]
    $i = 1
    foreach ($line in $prm_f) {
        Write-Host $i : $line

        $col = $line.Split(",")
        $FOLDER = "<" + $col[0] + ">"
        $VALUE = $col[1]
        Write-Host "FOLDER: $FOLDER VALUE: $VALUE"

        $file_contents = $(Get-Content -Encoding ascii ./$targetFile) -replace $FOLDER,$VALUE
        Write-output $file_contents | Out-File ./$targetFile -Encoding ascii

        $i++
    }

    return $RET_OK
}

#---------------------------------------------------
# Functions for configuring QMGR
#
# input:
# $TARGET_QMGR: Queue Manager Name
#---------------------------------------------------
function fnConfigQmgr
{
    fnExecCmd "./mqpcf mqsc -qm $TARGET_QMGR -wi 60 -f ./${TARGET_QMGR}.mqsc"
    return $RET_OK
}

#---------------------------------------------
# Functions for Checking QMGR Status
#
# input:
# $TARGET_QMGR: Queue Manager Name
#---------------------------------------------
function fnChkQmgrStatus
{
    $result = ./mqpcf qms -qm $TARGET_QMGR STATUS | Select-String -SimpleMatch "RUNNING" -quiet
    if ($result -ne $True)
    {
        return $RET_NG
    }

    return $RET_OK
}

#---------------------------------------------
# Functions for Checking CMD Server Status
#
# input:
# $TARGET_QMGR: Queue Manager Name
#---------------------------------------------
function fnChkCmdSvrStatus
{
    $result = ./mqpcf qms -qm $TARGET_QMGR CMDSERV | Select-String -SimpleMatch "CMDSERV(RUNNING)" -quiet
    if ($result -ne $True)
    {
        return $RET_NG
    }

    return $RET_OK
}

#-----------------------------------------------------------
# Functions for waiting QMGR and Command Server up
#
# input:
# $TARGET_QMGR: Queue Manager Name
# $QMGR_STARTUP_TIME: Queue Manager startup wait timer
# $CMDSVR_STARTUP_TIME: Command Server startup wait timer
#-----------------------------------------------------------
function fnWaitCSVRUp
{
    $retcd = fnChkQmgrStatus
    if ($retcd -ne $RET_OK)
    {
        Write-Host "Waiting queue manager startup $QMGR_STARTUP_TIME sec ..."
        Start-Sleep $QMGR_STARTUP_TIME
    }

    $retcd = fnChkCmdSvrStatus
    if ($retcd -ne $RET_OK)
    {
        Write-Host "Waiting command server startup $CMDSVR_STARTUP_TIME sec ..."
        Start-Sleep $CMDSVR_STARTUP_TIME
    }

    $retcd = fnChkCmdSvrStatus
    if ($retcd -ne $RET_OK)
    {
        Write-Host "The command server is not started."
        return $RET_NG
    }

    return $RET_OK
}

#----------------
# Program start
#----------------
if ($m -eq "")
{
    $script_name = $myInvocation.MyCommand.name
    Write-output "${script_name} -m <Qmgr> [-f <source mqsc file>]"
    exit(1)
}

$TARGET_QMGR = $m

# create MQSC source file
$retcd = fnReplaceParameters $f
if ( $retcd -ne $RET_OK )
{
    exit $RET_NG
}

# Create QMGR
$retcd = fnCreateQmgr
if ($retcd -ne $RET_OK)
{
    exit $RET_NG
}

# Start QNGR
$retcd = fnStartQmgr
if ($retcd -ne $RET_OK)
{
    exit $RET_NG
}

# Wait QMGR and Command Server Up
$retcd = fnWaitCSVRUp
if ($retcd -ne $RET_OK)
{
    exit $RET_NG
}

# Configure QMGR
$retcd = fnConfigQmgr
exit $retcd

