# Windows PwowerShell
#---------------------------------------------------------
# Run Performance Test
#
# Arguments:
# $1 : Queue manager name
# $2 : Test list file name
# $3 : Output file name
#
# History
#
# Date        Create/Update           Description
# 2022.02.21  Pulsar Integration Inc. Initial Version
#---------------------------------------------------------
Param(
    [String]$m = "",
    [String]$i = "",
    [String]$o = ""
)

# return code
[int]$RET_OK=0
[int]$RET_NG=1

# import library
. ./lib/libpfmts.ps1

# source environments for performance test
fnSourceEnvironments

#------------------------------------------------------------------
# Functions for Performance Tests
#
# input:
# $1: Queue Manager Name
# $2: Test List Name
# $3: Output File Name
#
# Test List CSV format:
# Colomn: Description
# 1: Test ID unique for each row
# 2: Number of shot
# 3: 0: Calculate average of all test shots
#    1: Calculate the average of the remaining test shots, excluding
#      the smallest and largest times (valid when the number of shots is
#      3 or more)
# 4: Workload Name
#    RR-1: MQI Local Bindings
#    RR-2: MQI Client Bindings
#    RR-5: MQI Distributed Queuing
#    PL-1: MQ Cluster
# 5: Persistent Mode
#    0: Non-Persistent
#    1: Persistent
# 6: The number of connections
# 7: The number of messages,
# 8: Message size
# 9: Input file Name
#------------------------------------------------------------------
function fnInvokePerformanceFunction
{
    param(
        [string]$TARGET_QMGR = "",
        [string]$TestListFile = "",
        [string]$OutputFile = "",
        [ref]$retcd = $RET_OK
    )

    $csv = (Import-Csv $TestListFile)
    foreach ($CsvLine in $csv)
    {
        if ($CsvLine."Test ID" -like "#*")
        {
            continue
        }

        Write-Host ""
        Write-Host ""
        Write-Host $CsvLine

        [string]$TestID = $CsvLine."Test ID"
        [int]$Shots = $CsvLine.Shots
        [int]$Mode = $CsvLine.Mode
        [string]$Workload = $CsvLine.Workload
        [int]$Persistent = $CsvLine.Persistent
        [int]$Connections = $CsvLine.Connections
        [int]$Messages = $CsvLine.Messages
        [int]$Bytes = $CsvLine.Bytes
        [string]$DataFile = $CsvLine.DataFile

        [string]$MQ_CONSTANT = "MQPER_NOT_PERSISTENT"
        if ( $Persistent -eq 1 )
        {
            $MQ_CONSTANT = "MQPER_PERSISTENT"
        }

        [string]$EXEC_SCRIPT = ""

        switch ( $Workload )
        {
          RR-1
          {
             $EXEC_SCRIPT = $env:RR1_SCRIPT
          }
          RR-1t
          {
             $EXEC_SCRIPT = $env:RR1_SCRIPT_THREAD
          }
          RR-2
          {
             $EXEC_SCRIPT = $env:RR2_SCRIPT
          }
          RR-2t
          {
             $EXEC_SCRIPT = $env:RR2_SCRIPT_THREAD
          }
          RR-5
          {
             $EXEC_SCRIPT = $env:RR5_SCRIPT
          }
          PL-1
          {
             $EXEC_SCRIPT = $env:PL1_SCRIPT
          }
          default
          {
              Write-output "Not Supported Workload: ${Workload}"
          }
        }

        $dateTime = Get-Date -Format "yyyyMMdd_HHmmss"
        $rootDirName = "logs\${TestID}_${dateTime}"
        # Create a directory in advance so that the first console message can be written
        mkdir $rootDirName

        for ($repeat = 0; $repeat -lt $Shots; $repeat++)
        {
            Write-Host ""
            Write-Host "TestID:" $TestID "Shot:" ($repeat + 1)

            Invoke-Expression "./${EXEC_SCRIPT} $TARGET_QMGR $Connections $Messages $Bytes $rootDirName $Workload $MQ_CONSTANT $DataFile"
            Start-Sleep 1
        }

#        if ($OutputFile -eq "")
#        {
#            fnGetAverageRoundTripsPerSec -ConsoleLog ${rootDirName}/console.log -Mode $Mode
#        }
#        else
#        {
            #fnGetAverageRoundTripsPerSec -ConsoleLog ${rootDirName}/console.log -Mode $Mode | Out-File $OutputFile -Append
            fnGetAverageRoundTripsPerSec -ConsoleLog ${rootDirName}/console.log -Mode $Mode | tee-object $OutputFile -Append
#        }
    }

    #return $RET_OK
    $retcd.Value = $RET_OK
}

#----------------
# Program start
#----------------
if ($m -eq "" -or $i -eq "" -or $o -eq "")
{
    $script_name = $myInvocation.MyCommand.name
    Write-output "${script_name} -m <Qmgr> -i <Test List Name> -o <Output File Name>"
    exit(1)
}

# Invoke Performance Function
[int]$retcd = $RET_OK
fnInvokePerformanceFunction $m $i $o ([ref]$retcd)
exit $retcd
